import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, Star, Calendar, Video, MessageSquare } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyBookings.css'

const MyBookings = () => {
  const [bookings, setBookings] = useState([])

  useEffect(() => {
    loadBookings()
    const interval = setInterval(loadBookings, 500)
    return () => clearInterval(interval)
  }, [])

  const loadBookings = () => {
    const saved = localStorage.getItem('studioBookings')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setBookings(parsed)
      } catch (e) {
        setBookings([])
      }
    } else {
      const defaultBookings = [
        {
          id: 1,
          studio: 'StudioSpace Premium',
          date: '2025-03-20',
          equipment: 'Профессиональное освещение, камеры',
          review: 'Отличная студия, все оборудование работает идеально!'
        },
        {
          id: 2,
          studio: 'StudioSpace Chroma',
          date: '2025-03-15',
          equipment: 'Хромакей, студийное освещение',
          review: 'Прекрасные условия для съемки'
        },
        {
          id: 3,
          studio: 'StudioSpace Interior',
          date: '2025-03-10',
          equipment: 'Интерьерная студия, естественный свет',
          review: 'Очень уютная атмосфера'
        }
      ]
      setBookings(defaultBookings)
      localStorage.setItem('studioBookings', JSON.stringify(defaultBookings))
    }
  }

  const totalBookings = bookings.length
  const totalStudios = new Set(bookings.map(b => b.studio)).size
  const averageRating = 4.8

  return (
    <div className="my-bookings">
      <div className="container">
        <div className="page-header">
          <motion.h1 
            className="page-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            Мои бронирования
          </motion.h1>
          <motion.p 
            className="page-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            История ваших аренд студий
          </motion.p>
        </div>

        <motion.div 
          className="stats-minimal"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{totalBookings}</div>
            <div className="stat-minimal-label">Бронирований</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{averageRating}</div>
            <div className="stat-minimal-label">Рейтинг</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{totalStudios}</div>
            <div className="stat-minimal-label">Студий</div>
          </div>
        </motion.div>

        <ImageSlider images={[
          '/images/studio-1.jpg',
          '/images/studio-2.jpg',
          '/images/studio-3.jpg',
          '/images/studio-4.jpg'
        ]} />

        <motion.section 
          className="bookings-list"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Ваши бронирования</h2>
          {bookings.length > 0 ? (
            <div className="bookings-vertical">
              {bookings.map((booking, index) => (
                <motion.div 
                  key={booking.id} 
                  className="booking-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ x: 4, transition: { duration: 0.2 } }}
                >
                  <div className="booking-item-left">
                    <div className="booking-item-icon">
                      <Camera size={24} strokeWidth={2} />
                    </div>
                    <div className="booking-item-content">
                      <div className="booking-item-header">
                        <h3 className="booking-item-studio">{booking.studio}</h3>
                        <div className="booking-item-rating">
                          {Array.from({ length: 5 }, (_, i) => (
                            <Star
                              key={i}
                              size={16}
                              fill={i < 4 ? '#F59E0B' : 'none'}
                              stroke={i < 4 ? '#F59E0B' : 'var(--text-light)'}
                              strokeWidth={1.5}
                            />
                          ))}
                        </div>
                      </div>
                      <div className="booking-item-details">
                        <div className="booking-item-detail">
                          <Calendar size={16} strokeWidth={2} />
                          <span>
                            {new Date(booking.date).toLocaleDateString('ru-RU', { 
                              day: 'numeric', 
                              month: 'long', 
                              year: 'numeric' 
                            })}
                          </span>
                        </div>
                        <div className="booking-item-detail">
                          <Video size={16} strokeWidth={2} />
                          <span>{booking.equipment}</span>
                        </div>
                        {booking.review && (
                          <div className="booking-item-review">
                            <MessageSquare size={16} strokeWidth={2} />
                            <span>{booking.review}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Camera size={64} strokeWidth={1.5} className="empty-icon" />
              <h3 className="empty-title">Нет бронирований</h3>
              <p className="empty-text">Забронируйте студию для вашего проекта</p>
            </div>
          )}
        </motion.section>

      </div>
    </div>
  )
}

export default MyBookings

